#!/bin/sh

# Ensure the script is run as root
if [ "$(id -u)" -ne 0 ]; then
    echo "This script must be run as root (use sudo to execute it)." >&2
    exit 1
fi

SERVICE_DIR="$(cd "$(dirname "$0")" || exit 1; pwd)"
SERVICES="cpl360_service_manager cpl360_totp_service"

# Ensure systemctl is available
if ! command -v systemctl >/dev/null 2>&1; then
    echo "Error: systemctl not found. This script requires systemd." >&2
    exit 1
fi

for SERVICE in $SERVICES; do
    echo "Stopping service: $SERVICE.service..."
    if ! sudo systemctl stop "$SERVICE.service"; then
        echo "Error: Failed to stop service $SERVICE.service." >&2
        exit 1
    fi
done

# Wait for services to stop completely
sleep 2

# Check if any service is still active
for SERVICE in $SERVICES; do
    if systemctl is-active --quiet "$SERVICE.service"; then
        echo "Warning: Service $SERVICE.service is still running!" >&2
        exit 1
    fi
done

echo "All specified services have been stopped successfully."

# Optional: Show final status (useful for debugging)
for SERVICE in $SERVICES; do
    echo "Checking status of service: $SERVICE.service"
    systemctl status --no-pager "$SERVICE.service"
done