#!/bin/sh

# Check if an input parameter is provided
if [ -z "$1" ]; then
    echo "Usage: $0 <keyword>"
    echo "Please provide a keyword as the argument, e.g., tpi_01"
    exit 1
fi

# Receive the input parameter as the keyword
keyword="$1"

# Set the log file name
script_dir=$(dirname "$(realpath "$0")")
log_file="$script_dir/process_kill.log"

# Get the current timestamp
current_time=$(date +"%Y-%m-%d %H:%M:%S")

# Log the start of the search for the keyword
echo "[$current_time] Searching for ffmpeg processes with keyword: '$keyword'" >> "$log_file"

# Use pgrep to search for running ffmpeg processes
processes=$(pgrep -af ffmpeg)

# If no ffmpeg processes are found
if [ -z "$processes" ]; then
    echo "No running ffmpeg processes found." | tee -a "$log_file"
    exit 0
fi

# Use a regular expression to match the URL
matching_process=$(echo "$processes" | grep -E "rtsp://localhost:8554/${keyword}(-webrtc)?$")

# If no matching process is found
if [ -z "$matching_process" ]; then
    echo "No ffmpeg process found with the exact URL 'rtsp://localhost:8554/$keyword'." | tee -a "$log_file"
    exit 0
fi

# Display and log the matching process
echo "[$current_time] Matching process found: $matching_process" >> "$log_file"

# Extract the PID and attempt to terminate the process
pid=$(echo "$matching_process" | awk '{print $1}')
echo "[$current_time] Attempting to gracefully terminate process PID: $pid" >> "$log_file"
kill "$pid"

# Wait for the process to terminate
sleep 0.2

# Check if the process is still running
if kill -0 "$pid" 2>/dev/null; then
    echo "[$current_time] Process PID: $pid is unresponsive, attempting to force terminate..." >> "$log_file"
    if kill -9 "$pid"; then
        echo "[$current_time] Process $pid has been forcefully terminated." >> "$log_file"
    else
        echo "[$current_time] Failed to force terminate process $pid." >> "$log_file"
    fi
else
    echo "[$current_time] Process $pid has been successfully terminated." >> "$log_file"
fi

echo "[$current_time] Task completed for keyword: '$keyword'" >> "$log_file"