#!/bin/sh

# Ensure the script is run as root
if [ "$(id -u)" -ne 0 ]; then
    echo "This script must be run as root (use sudo to execute it)." >&2
    exit 1
fi

SERVICE_DIR="$(cd "$(dirname "$0")" || exit 1; pwd)"
SYSTEMD_PATH="/etc/systemd/system"
SERVICES="cpl360_service_manager cpl360_totp_service"

# Ensure systemctl is available
if ! command -v systemctl >/dev/null 2>&1; then
    echo "Error: systemctl not found. This script requires systemd." >&2
    exit 1
fi

for SERVICE in $SERVICES; do
    echo "Stopping service: $SERVICE.service..."
    sudo systemctl stop "$SERVICE.service"

    # Verify if the service is stopped
    if sudo systemctl is-active --quiet "$SERVICE.service"; then
        echo "Warning: Service $SERVICE.service is still running! You may need to manually stop it." >&2
    fi

    # Disable service if it's enabled
    if sudo systemctl is-enabled --quiet "$SERVICE.service"; then
        echo "Disabling service: $SERVICE.service..."
        sudo systemctl disable "$SERVICE.service"
    else
        echo "Service $SERVICE.service is already disabled."
    fi

    # Remove the service file
    if [ -f "$SYSTEMD_PATH/$SERVICE.service" ]; then
        echo "Removing service file: $SYSTEMD_PATH/$SERVICE.service..."
        sudo rm -f "$SYSTEMD_PATH/$SERVICE.service"
    else
        echo "Service file $SYSTEMD_PATH/$SERVICE.service not found. Skipping removal."
    fi
done

# Reload systemd daemon and clear failed status
echo "Reloading systemd daemon..."
sudo systemctl daemon-reload

for SERVICE in $SERVICES; do
    sudo systemctl reset-failed "$SERVICE.service"
done

echo "All specified services have been removed successfully."