#!/bin/sh

# Ensure the script is run as root
if [ "$(id -u)" -ne 0 ]; then
    echo "This script must be run as root (use sudo to execute it)." >&2
    exit 1
fi

SCRIPT_DIR="$(cd "$(dirname "$0")" || exit 1; pwd)"
SCRIPT_PARENT_DIR="$(dirname "$SCRIPT_DIR")"

if [ ! -d "$SCRIPT_PARENT_DIR/service" ]; then
    echo "Error: Service directory not found: $SCRIPT_PARENT_DIR/service" >&2
    exit 1
fi
SERVICE_DIR="$(cd "$SCRIPT_PARENT_DIR/service" && pwd)"

SERVICES="cpl360_service_manager cpl360_totp_service"
SYSTEMD_DIR="/etc/systemd/system"

# Ensure systemctl is available
if ! command -v systemctl >/dev/null 2>&1; then
    echo "Error: systemctl not found. This script requires systemd." >&2
    exit 1
fi

# Install each service
for SERVICE in $SERVICES; do
    TEMPLATE_FILE="${SERVICE_DIR}/${SERVICE}.service.template"
    GENERATED_FILE="${SERVICE_DIR}/${SERVICE}.service"

    if [ ! -f "${TEMPLATE_FILE}" ]; then
        echo "Error: Template file not found: ${TEMPLATE_FILE}" >&2
        exit 1
    fi

    sed "s|{{WORKING_DIRECTORY}}|${SCRIPT_PARENT_DIR}|g" "${TEMPLATE_FILE}" > "${GENERATED_FILE}"

    if [ ! -f "${GENERATED_FILE}" ]; then
        echo "Error: Failed to generate service file: ${GENERATED_FILE}" >&2
        exit 1
    fi

    sudo mv -f "${GENERATED_FILE}" "${SYSTEMD_DIR}/${SERVICE}.service" || {
        echo "Error: Failed to move service file to ${SYSTEMD_DIR}" >&2
        exit 1
    }

    echo "Enabling service: ${SERVICE}.service for startup"
    if ! sudo systemctl enable "${SERVICE}.service"; then
        echo "Error: Failed to enable service ${SERVICE}.service for startup." >&2
        exit 1
    fi
done

# Reload systemd daemon
sudo systemctl daemon-reload || {
    echo "Error: Failed to reload systemd daemon." >&2
    exit 1
}

echo "The service files have been successfully installed and systemd has been reloaded."

# Ask if the user wants to start the services now
read -p "Do you want to start the services now? (y/N): " RESPONSE
if [ "$RESPONSE" = "y" ] || [ "$RESPONSE" = "Y" ]; then
    for SERVICE in $SERVICES; do
        echo "Starting service: ${SERVICE}.service"
        if sudo systemctl start "${SERVICE}.service"; then
            echo "Service ${SERVICE}.service has been started successfully."
        else
            echo "Error: Failed to start service ${SERVICE}.service." >&2
            exit 1
        fi
    done
else
    echo "You chose not to start the services. You can start them later by running './start_service.sh'."
    echo "The installed services are: ${SERVICES[*]}"
fi